﻿// NILTOの一覧レスポンス形式。
type NiltoListResponse<T> = {
  total: number;
  limit: number;
  offset: number;
  data: T[];
};

// 画像フィールドの最小形（配列で返る場合がある）。
type NiltoImage = {
  url?: string;
  alt?: string;
  width?: number;
  height?: number;
};

// 著者参照モデル。
type NiltoAuthor = {
  name?: string;
  profile?: string;
};

// 記事モデル（テンプレ「シンプルブログ」想定）。
export type Article = {
  _id: string;
  _published_at?: string;
  _created_at?: string;
  title?: string;
  category?: string | { name?: string; title?: string; slug?: string };
  image?: NiltoImage | NiltoImage[] | null;
  body?: string | { html?: string } | null;
  author?: NiltoAuthor | null;
};

// APIキーは環境変数（.envファイル）で設定。
const API_BASE_URL = "https://cms-api.nilto.com/v1";
const API_KEY = process.env.NILTO_API_KEY;
const LANG = "ja";

// クエリ付きURLを生成。
function buildUrl(path: string, params: Record<string, string>): string {
  const searchParams = new URLSearchParams(params);
  return `${API_BASE_URL}${path}?${searchParams.toString()}`;
}

// ステータス別にユーザー向けのメッセージへ変換。
function resolveErrorMessage(status: number): string {
  if (status === 401 || status === 403) {
    return `APIが ${status} です`;
  }
  return `APIエラー (${status}) です`;
}

// 画面に分かりやすいエラーを出すための共通フェッチ。
async function fetchFromNilto<T>(url: string): Promise<T> {
  if (!API_KEY) {
    throw new Error("NILTO_API_KEY が未設定です");
  }

  const res = await fetch(url, {
    headers: {
      "X-NILTO-API-KEY": API_KEY,
    },
    cache: "no-store",
  });

  if (!res.ok) {
    throw new Error(resolveErrorMessage(res.status));
  }

  return res.json() as Promise<T>;
}

// トップページ用の記事一覧。
export async function fetchArticles(): Promise<Article[]> {
  // 一覧は著者参照を含め、本文はHTMLで取得する。
  const url = buildUrl("/contents", {
    model: "article",
    lang: LANG,
    depth: "1",
    limit: "20",
    "body[format]": "html",
    select: "title,image,category,author,body,_id,_published_at,_created_at",
  });

  const response = await fetchFromNilto<NiltoListResponse<Article>>(url);
  return response.data || [];
}

// 記事詳細（content_id指定）。
export async function fetchArticleById(id: string): Promise<Article | null> {
  // 詳細も本文はHTMLで取得する。
  const url = buildUrl(`/contents/${id}`, {
    lang: LANG,
    depth: "1",
    "body[format]": "html",
  });

  return fetchFromNilto<Article>(url);
}

export function getImageUrl(image?: Article["image"]): string | null {
  // 画像が配列でも単体でも扱えるようにする。
  if (!image) return null;
  if (Array.isArray(image)) {
    return image[0]?.url || null;
  }
  return image.url || null;
}

const PLACEHOLDER_IMAGE_URL =
  "data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='1200' height='800' viewBox='0 0 1200 800'><rect width='1200' height='800' fill='%23f1ede6'/><text x='50%' y='50%' dominant-baseline='middle' text-anchor='middle' font-family='sans-serif' font-size='48' fill='%23909090'>No Image</text></svg>";

export function getImageUrlOrPlaceholder(image?: Article["image"]): string {
  // 画像がない場合はプレースホルダーを返す。
  return getImageUrl(image) || PLACEHOLDER_IMAGE_URL;
}

export function getBodyHtml(body?: Article["body"]): string {
  // HTML文字列のみ返す（存在しなければ空）。
  if (!body) return "";
  if (typeof body === "string") return body;
  return body.html || "";
}

export function getCategoryLabel(category?: Article["category"]): string {
  // 文字列/参照オブジェクトのどちらでも表示名を返す。
  if (!category) return "";
  if (typeof category === "string") return category;
  return category.name || category.title || category.slug || "";
}

export function formatDate(value?: string): string {
  // 表示用に短い日付へ整形。
  if (!value) return "";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return value;
  return new Intl.DateTimeFormat("ja-JP", {
    year: "numeric",
    month: "short",
    day: "2-digit",
  }).format(date);
}
